% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getMappedEntrezIDs.R
\name{getMappedEntrezIDs}
\alias{getMappedEntrezIDs}
\title{Get mapped Entrez Gene IDs from CpG probe names}
\usage{
getMappedEntrezIDs(
  sig.cpg,
  all.cpg = NULL,
  array.type = c("450K", "EPIC", "EPIC_V2"),
  anno = NULL,
  genomic.features = c("ALL", "TSS200", "TSS1500", "Body", "1stExon", "3'UTR", "5'UTR",
    "ExonBnd")
)
}
\arguments{
\item{sig.cpg}{Character vector of significant CpG sites used for testing
gene set enrichment.}

\item{all.cpg}{Character vector of all CpG sites tested. Defaults to all CpG
sites on the array.}

\item{array.type}{The Illumina methylation array used. Options are "450K", 
"EPIC" or "EPIC_V2".}

\item{anno}{Optional. A \code{DataFrame} object containing the complete
array annotation as generated by the \code{\link{minfi}}
\code{\link{getAnnotation}} function. Speeds up execution, if provided.}

\item{genomic.features}{Character vector or scalar indicating whether the 
gene set enrichment analysis should be restricted to CpGs from specific 
genomic locations. Options are "ALL", "TSS200","TSS1500","Body","1stExon",
"3'UTR","5'UTR","ExonBnd"; and the user can select any combination. Defaults
to "ALL".}
}
\value{
A list with the following elements \item{sig.eg}{ mapped Entrez Gene
IDs for the significant probes } \item{universe}{ mapped Entrez Gene IDs for
all probes on the array, or for all the CpG probes tested.  } \item{freq}{
table output with numbers of probes associated with each gene } 
\item{equiv}{ table output with equivalent numbers of probes associated
with each gene taking into account multi-gene bias} \item{de}{ a
vector of ones and zeroes of the same length of universe indicating which
genes in the universe are significantly differentially methylated.  }
\item{fract.counts}{ a dataframe with 2 columns corresponding to the Entrez
Gene IDS for the significant probes and the associated weight to account for
multi-gene probes.  }
}
\description{
Given a set of CpG probe names and optionally all the CpG sites tested, this
function outputs a list containing the mapped Entrez Gene IDs as well as the
numbers of probes per gene, and a vector indicating significance.
}
\details{
This function is used by the gene set testing functions \code{gometh} and
\code{gsameth}. It maps the significant CpG probe names to Entrez Gene IDs,
as well as all the CpG sites tested. It also calculates the numbers of
probes for gene. Input CpGs are able to be restricted by genomic features 
using the \code{genomic.features} argument.

Genes associated with each CpG site are obtained from the annotation package
\code{IlluminaHumanMethylation450kanno.ilmn12.hg19} if the array type is
"450K". For the EPIC array, the annotation package
\code{IlluminaHumanMethylationEPICanno.ilm10b4.hg19} is used. For the EPIC v2 
array, the annotation package 
\code{IlluminaHumanMethylationEPICv2anno.20a1.hg38} is used. To use a
different annotation package, please supply it using the \code{anno}
argument.
}
\examples{

\dontrun{ # to avoid timeout on Bioconductor build
library(IlluminaHumanMethylation450kanno.ilmn12.hg19)
library(org.Hs.eg.db)
library(limma)
ann <- getAnnotation(IlluminaHumanMethylation450kanno.ilmn12.hg19)

# Randomly select 1000 CpGs to be significantly differentially methylated
sigcpgs <- sample(rownames(ann),1000,replace=FALSE)

# All CpG sites tested
allcpgs <- rownames(ann)

mappedEz <- getMappedEntrezIDs(sigcpgs,allcpgs,array.type="450K")
names(mappedEz)
# Entrez IDs of the significant genes
mappedEz$sig.eg[1:10]
# Entrez IDs for the universe
mappedEz$universe[1:10]
# Number of CpGs per gene
mappedEz$freq[1:10]
# Equivalent numbers of CpGs measured per gene
mappedEz$equiv[1:10]
A vector of 0s and 1s indicating which genes in the universe are significant
mappedEz$de[1:10]
}

}
\seealso{
\code{\link{gometh},\link{gsameth}}
}
\author{
Belinda Phipson
}
