% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_array_coords.R
\name{add_array_coords}
\alias{add_array_coords}
\title{Add transformed array and pixel coordinates to a \code{SpatialExperiment}}
\usage{
add_array_coords(
  spe,
  sample_info,
  coords_dir,
  calc_error_metrics = FALSE,
  algorithm = c("LSAP", "Euclidean")
)
}
\arguments{
\item{spe}{A
\link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class} object.}

\item{sample_info}{A \code{data.frame()} with columns \code{capture_area},
\code{group}, \code{fiji_xml_path}, \code{fiji_image_path},
\code{spaceranger_dir}, \code{intra_group_scalar}, and
\code{group_hires_scalef}. The last two are made by \code{rescale_fiji_inputs()}.}

\item{coords_dir}{A \code{character(1)} vector giving the directory
containing sample directories each with \code{tissue_positions.csv},
\code{scalefactors_json.json}, and \code{tissue_lowres_image.png} files
produced from refinement with \link[=prep_fiji_coords]{prep_fiji_coords()}
and related functions.}

\item{calc_error_metrics}{A \code{logical(1)} vector indicating whether to
calculate error metrics related to mapping spots to well-defined array
coordinates. If \code{TRUE}, adds \code{euclidean_error} and
\code{shared_neighbors} spot-level metrics to the \code{colData()}. The former
indicates distance in number of inter-spot distances to "move" a spot to the
new array position; the latter indicates the fraction of neighbors for the
associated capture area that are retained after mapping, which can be quite
time-consuming to compute.}

\item{algorithm}{A \code{character(1)} vector indicating which mapping
algorithm to employ when computing group-wide array coordinates. The default of
"LSAP" is generally recommended, as it guarantees one-to-one mappings
at the cost of computational time and some Euclidean error. The faster
alternative "Euclidean" minimizes Euclidean error but may produce duplicate
mappings, which is generally undesirable downstream (for clustering, etc).}
}
\value{
A \link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class}
object with additional \code{colData}
columns \code{pxl_row_in_fullres_[suffix]} and \code{pxl_col_in_fullres_[suffix]}
with \code{[suffix]} values \code{original} and \code{rounded};
\code{array_row_original} and \code{array_col_original} columns; and
modified \code{colData()} columns \code{array_row} and
\code{array_col} and \code{spatialCoords()} with their transformed values.
}
\description{
Given a \link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class},
sample information, and coordinates
produced from the refinement workflow, add array and pixel coordinates
appropriate for the linearly transformed capture areas making up each group
present in the \link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class}.
}
\details{
Array coordinates are determined via an algorithm that fits each spot to
the nearest spot on a new, imaginary, Visium-like capture area. The imaginary
capture area differs from a real capture area only in its extent; array
coordinates still start at 0 but may extend arbitrarily beyond the normal
maximum indices of 77 and 127 to fit every capture area in each group
defined in the \link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class}.
The goal is to return well-defined
array coordinates in a consistent spatial orientation for each group, such
that downstream applications, such as clustering with \code{BayesSpace}, can
process each group as if it really were one capture area in the first place.
See
\url{https://research.libd.org/visiumStitched/articles/visiumStitched.html#defining-array-coordinates}
for more details.
}
\examples{
if (!exists("spe")) {
    spe <- spatialLIBD::fetch_data(type = "visiumStitched_brain_spe")
}

########################################################################
#   Prepare sample_info
########################################################################

sample_info <- dplyr::tibble(
    group = "Br2719",
    capture_area = c("V13B23-283_A1", "V13B23-283_C1", "V13B23-283_D1")
)
#   Add 'spaceranger_dir' column
sr_dir <- tempdir()
temp <- unzip(
    spatialLIBD::fetch_data("visiumStitched_brain_spaceranger"),
    exdir = sr_dir
)
sample_info$spaceranger_dir <- file.path(
    sr_dir, sample_info$capture_area, "outs", "spatial"
)

#   Add Fiji-output-related columns
fiji_dir <- tempdir()
temp <- unzip(
    spatialLIBD::fetch_data("visiumStitched_brain_Fiji_out"),
    exdir = fiji_dir
)
sample_info$fiji_xml_path <- temp[grep("xml$", temp)]
sample_info$fiji_image_path <- temp[grep("png$", temp)]

## Re-size images and add more information to the sample_info
sample_info <- rescale_fiji_inputs(sample_info, out_dir = tempdir())

## Preparing Fiji coordinates and images for build_SpatialExperiment()
spe_input_dir <- tempdir()
prep_fiji_coords(sample_info, out_dir = spe_input_dir)
prep_fiji_image(sample_info, out_dir = spe_input_dir)

########################################################################
#   Add array coordinates
########################################################################

#   Run with Euclidean algorithm for speed. On real analyses, "LSAP" is
#   generally recommended.
spe_new <- add_array_coords(
    spe, sample_info, tempdir(), algorithm = "Euclidean"
)

#    Several columns related to spatial coordinates were added
added_cols_regex <- "^(array|pxl)_(row|col)(_in_fullres)?_(original|rounded)$"
colnames(SummarizedExperiment::colData(spe_new))[
    grep(added_cols_regex, colnames(SummarizedExperiment::colData(spe_new)))
]

#    'array_row', 'array_col', and spatialCoords() were overwritten with
#    their transformed values
head(spe$array_row)
head(spe$array_col)
head(SpatialExperiment::spatialCoords(spe_new))
}
\author{
Nicholas J. Eagles
}
