% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_findChromPeaks-functions.R
\name{peaksWithCentWave}
\alias{peaksWithCentWave}
\title{Identify peaks in chromatographic data using centWave}
\usage{
peaksWithCentWave(
  int,
  rt,
  peakwidth = c(20, 50),
  snthresh = 10,
  prefilter = c(3, 100),
  integrate = 1,
  fitgauss = FALSE,
  noise = 0,
  verboseColumns = FALSE,
  firstBaselineCheck = TRUE,
  extendLengthMSW = FALSE,
  ...
)
}
\arguments{
\item{int}{\code{numeric} with intensity values.}

\item{rt}{\code{numeric} with the retention time for the intensities. Length has
to be equal to \code{length(int)}.}

\item{peakwidth}{\code{numeric(2)} with the lower and upper bound of the
expected peak width.}

\item{snthresh}{\code{numeric(1)} defining the signal to noise ratio cutoff.
Peaks with a signal to noise ratio < \code{snthresh} are omitted.}

\item{prefilter}{\code{numeric(2)} (\code{c(k, I)}): only regions of interest with at
least \code{k} centroids with signal \verb{>= I} are returned in the first
step.}

\item{integrate}{\code{numeric(1)}, integration method. For \code{integrate = 1} peak
limits are found through descending on the mexican hat filtered data,
for \code{integrate = 2} the descend is done on the real data. The latter
method is more accurate but prone to noise, while the former is more
robust, but less exact.}

\item{fitgauss}{\code{logical(1)} whether or not a Gaussian should be fitted
to each peak.}

\item{noise}{\code{numeric(1)} defining the minimum required intensity for
centroids to be considered in the first analysis step (definition of
the \emph{regions of interest}).}

\item{verboseColumns}{\code{logical(1)}: whether additional peak meta data
columns should be returned.}

\item{firstBaselineCheck}{\code{logical(1)}. If \code{TRUE} continuous data within
regions of interest is checked to be above the first baseline. In detail,
a first \emph{rough} estimate of the noise is calculated and peak detection
is performed only in regions in which multiple sequential signals are
higher than this first estimated baseline/noise level.}

\item{extendLengthMSW}{\code{logical(1)}. If \code{TRUE} the "open" method of EIC
extension is used, rather than the default "reflect" method.}

\item{...}{currently ignored.}
}
\value{
A matrix, each row representing an identified chromatographic peak, with
columns:
\itemize{
\item \code{"rt"}: retention time of the peak's midpoint (time of the maximum signal).
\item \code{"rtmin"}: minimum retention time of the peak.
\item \code{"rtmax"}: maximum retention time of the peak.
\item \code{"into"}: integrated (original) intensity of the peak.
\item \code{"intb"}: per-peak baseline corrected integrated peak intensity.
\item \code{"maxo"}: maximum (original) intensity of the peak.
\item \code{"sn"}: signal to noise ratio of the peak defined as
\code{(maxo - baseline)/sd} with \code{sd} being the standard deviation of the local
chromatographic noise.
}

Additional columns for \code{verboseColumns = TRUE}:
\itemize{
\item \code{"mu"}: gaussian parameter mu.
\item \code{"sigma"}: gaussian parameter sigma.
\item \code{"h"}: gaussian parameter h.
\item \code{"f"}: region number of the m/z ROI where the peak was localized.
\item \code{"dppm"}: m/z deviation of mass trace across scans in ppm (always \code{NA}).
\item \code{"scale"}: scale on which the peak was localized.
\item \code{"scpos"}: peak position found by wavelet analysis (index in \code{int}).
\item \code{"scmin"}: left peak limit found by wavelet analysis (index in \code{int}).
\item \code{"scmax"}: right peak limit found by wavelet analysis (index in \code{int}).
}
}
\description{
\code{peaksWithCentWave} identifies (chromatographic) peaks in purely
chromatographic data, i.e. based on intensity and retention time values
without m/z values.
}
\details{
The method uses the same algorithm for the peak detection than \link{centWave},
employs however a different approach to identify the initial regions in
which the peak detection is performed (i.e. the \emph{regions of interest} ROI).
The method first identifies all local maxima in the chromatographic data and
defines the corresponding positions +/- \code{peakwidth[2]} as the ROIs. Noise
estimation bases also on these ROIs and can thus be different from \link{centWave}
resulting in different signal to noise ratios.
}
\examples{

## Reading a file
library(MsExperiment)
library(xcms)
od <- readMsExperiment(system.file("cdf/KO/ko15.CDF", package = "faahKO"))

## Extract chromatographic data for a small m/z range
mzr <- c(272.1, 272.2)
chr <- chromatogram(od, mz = mzr, rt = c(3000, 3300))[1, 1]

int <- intensity(chr)
rt <- rtime(chr)

## Plot the region
plot(chr, type = "h")

## Identify peaks in the chromatographic data
pks <- peaksWithCentWave(intensity(chr), rtime(chr))
pks

## Highlight the peaks
rect(xleft = pks[, "rtmin"], xright = pks[, "rtmax"],
    ybottom = rep(0, nrow(pks)), ytop = pks[, "maxo"], col = "#ff000040",
    border = "#00000040")
}
\seealso{
\link{centWave} for a detailed description of the peak detection
method.

Other peak detection functions for chromatographic data: 
\code{\link{peaksWithMatchedFilter}()}
}
\author{
Johannes Rainer
}
\concept{peak detection functions for chromatographic data}
