% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gsvaNewAPI.R
\name{gsva}
\alias{gsva}
\alias{gsva,plageParam-method}
\alias{gsva,zscoreParam-method}
\alias{gsva,ssgseaParam-method}
\alias{gsva,gsvaParam-method}
\title{Gene Set Variation Analysis}
\usage{
\S4method{gsva}{plageParam}(param, verbose = TRUE, BPPARAM = SerialParam(progressbar = verbose))

\S4method{gsva}{zscoreParam}(param, verbose = TRUE, BPPARAM = SerialParam(progressbar = verbose))

\S4method{gsva}{ssgseaParam}(param, verbose = TRUE, BPPARAM = SerialParam(progressbar = verbose))

\S4method{gsva}{gsvaParam}(param, verbose = TRUE, BPPARAM = SerialParam(progressbar = verbose))
}
\arguments{
\item{param}{A parameter object of one of the following classes:
\itemize{
\item A \code{\link{gsvaParam}} object built using the constructor function
\code{\link{gsvaParam}}.
This object will trigger \code{gsva()} to use the GSVA algorithm by
Hänzelmann et al. (2013).
\item A \code{\link{plageParam}} object built using the constructor function
\code{\link{plageParam}}.
This object will trigger \code{gsva()} to use the PLAGE algorithm by
Tomfohr et al. (2005).
\item A \code{\link{zscoreParam}} object built using the constructor function
\code{\link{zscoreParam}}.
This object will trigger \code{gsva()} to use the combined z-score algorithm by
Lee et al. (2008).
\item A \code{\link{ssgseaParam}} object built using the constructor function
\code{\link{ssgseaParam}}.
This object will trigger \code{gsva()} to use the ssGSEA algorithm by
Barbie et al. (2009).
}}

\item{verbose}{Gives information about each calculation step. Default: \code{TRUE}.}

\item{BPPARAM}{An object of class \code{BiocParallelParam} specifying parameters
related to the parallel execution of some of the tasks and calculations
within this function.}
}
\value{
A gene-set by sample matrix of GSVA enrichment scores stored in a
container object of the same type as the input expression data container. If
the input was a base matrix or a \code{dgCMatrix} object, then the output will
be a base matrix object with the gene sets employed in the calculations
stored in an attribute called \code{geneSets}. If the input was an
\code{ExpressionSet} object, then the output will be also an \code{ExpressionSet}
object with the gene sets employed in the calculations stored in an
attribute called \code{geneSets}. If the input was an object of one of the
classes described in \code{\link{GsvaExprData}}, such as a \code{SingleCellExperiment},
then the output will be of the same class, where enrichment scores will be
stored in an assay called \code{es} and the gene sets employed in the
calculations will be stored in the \code{rowData} slot of the object under the
column name \code{gs}.
}
\description{
Estimates GSVA enrichment scores.
}
\examples{
library(GSVA)
library(limma)

p <- 10 ## number of genes
n <- 30 ## number of samples
nGrp1 <- 15 ## number of samples in group 1
nGrp2 <- n - nGrp1 ## number of samples in group 2

## consider three disjoint gene sets
geneSets <- list(set1=paste("g", 1:3, sep=""),
                 set2=paste("g", 4:6, sep=""),
                 set3=paste("g", 7:10, sep=""))

## sample data from a normal distribution with mean 0 and st.dev. 1
y <- matrix(rnorm(n*p), nrow=p, ncol=n,
            dimnames=list(paste("g", 1:p, sep="") , paste("s", 1:n, sep="")))

## genes in set1 are expressed at higher levels in the last 'nGrp1+1' to 'n' samples
y[geneSets$set1, (nGrp1+1):n] <- y[geneSets$set1, (nGrp1+1):n] + 2

## build design matrix
design <- cbind(sampleGroup1=1, sampleGroup2vs1=c(rep(0, nGrp1), rep(1, nGrp2)))

## fit linear model
fit <- lmFit(y, design)

## estimate moderated t-statistics
fit <- eBayes(fit)

## genes in set1 are differentially expressed
topTable(fit, coef="sampleGroup2vs1")

## build GSVA parameter object
gsvapar <- gsvaParam(y, geneSets)

## estimate GSVA enrichment scores for the three sets
gsva_es <- gsva(gsvapar)

## fit the same linear model now to the GSVA enrichment scores
fit <- lmFit(gsva_es, design)

## estimate moderated t-statistics
fit <- eBayes(fit)

## set1 is differentially expressed
topTable(fit, coef="sampleGroup2vs1")
}
\references{
Barbie, D.A. et al. Systematic RNA interference reveals that
oncogenic KRAS-driven cancers require TBK1.
\emph{Nature}, 462(5):108-112, 2009.
\doi{10.1038/nature08460}

Hänzelmann, S., Castelo, R. and Guinney, J. GSVA: Gene set
variation analysis for microarray and RNA-Seq data.
\emph{BMC Bioinformatics}, 14:7, 2013.
\doi{10.1186/1471-2105-14-7}

Lee, E. et al. Inferring pathway activity toward precise
disease classification.
\emph{PLoS Comp Biol}, 4(11):e1000217, 2008.
\doi{10.1371/journal.pcbi.1000217}

Tomfohr, J. et al. Pathway level analysis of gene expression
using singular value decomposition.
\emph{BMC Bioinformatics}, 6:225, 2005.
\doi{10.1186/1471-2105-6-225}
}
\seealso{
\code{\link{plageParam}}, \code{\link{zscoreParam}}, \code{\link{ssgseaParam}}, \code{\link{gsvaParam}},
\code{\link[BiocParallel:BiocParallelParam-class]{BiocParallelParam}},
\code{\link[Matrix:dgCMatrix-class]{dgCMatrix}},
\code{\link[Biobase]{ExpressionSet}},
\code{\link[SingleCellExperiment:SingleCellExperiment]{SingleCellExperiment}}
}
